<?php
// File: src/Controller/DonationsController.php
namespace App\Controller;

use Cake\Controller\Controller;
use Stripe\Stripe;
use Stripe\Checkout\Session;
use Stripe\Exception\ApiErrorException;

class DonationsController extends Controller
{
    public function initialize(): void
    {
        parent::initialize();
        // Disable security component for Stripe webhook
        $this->Components->Security->setConfig('validatePost', false);
    }

    // Donation form page
    public function index()
    {
        // Render donation form view
    }

    // Create Stripe Checkout Session
    public function createDonationSession()
    {
        // Set your Stripe secret key
        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

        try {
            // Validate and sanitize donation amount
            $donationAmount = $this->request->getData('amount');
            $donationAmount = floatval($donationAmount);

            // Create Stripe Checkout Session
            $session = Session::create([
                'payment_method_types' => ['card'],
                'line_items' => [[
                    'price_data' => [
                        'currency' => 'usd',
                        'unit_amount' => $donationAmount * 100, // Convert to cents
                        'product_data' => [
                            'name' => 'Donation',
                        ],
                    ],
                    'quantity' => 1,
                ]],
                'mode' => 'payment',
                'success_url' => $this->Url->build([
                    'controller' => 'Donations',
                    'action' => 'success'
                ], ['_full' => true]),
                'cancel_url' => $this->Url->build([
                    'controller' => 'Donations',
                    'action' => 'cancel'
                ], ['_full' => true]),
            ]);

            // Redirect to Stripe Checkout
            return $this->redirect($session->url);
        } catch (ApiErrorException $e) {
            // Log error and show user-friendly message
            $this->log($e->getMessage(), 'error');
            $this->Flash->error('Payment processing failed. Please try again.');
            return $this->redirect(['action' => 'index']);
        }
    }

    // Successful donation handler
    public function success()
    {
        $this->Flash->success('Thank you for your donation!');
        return $this->redirect(['action' => 'index']);
    }

    // Cancelled donation handler
    public function cancel()
    {
        $this->Flash->warning('Donation was cancelled.');
        return $this->redirect(['action' => 'index']);
    }

    // Stripe Webhook for donation confirmation
    public function webhook()
    {
        Stripe::setApiKey(env('STRIPE_SECRET_KEY'));

        $payload = file_get_contents('php://input');
        $sigHeader = $_SERVER['HTTP_STRIPE_SIGNATURE'];
        $endpointSecret = env('STRIPE_WEBHOOK_SECRET');

        try {
            $event = \Stripe\Webhook::constructEvent(
                $payload,
                $sigHeader,
                $endpointSecret
            );

            // Handle specific event types
            switch ($event->type) {
                case 'payment_intent.succeeded':
                    $paymentIntent = $event->data->object;
                    $this->_processSuccessfulDonation($paymentIntent);
                    break;

                // Add more event handlers as needed
            }

            return $this->response->withStatus(200);
        } catch(\UnexpectedValueException $e) {
            // Invalid payload
            return $this->response->withStatus(400);
        } catch(\Stripe\Exception\SignatureVerificationException $e) {
            // Invalid signature
            return $this->response->withStatus(400);
        }
    }

    // Internal method to process donation after payment
    protected function _processSuccessfulDonation($paymentIntent)
    {
        // Log donation details
        $amount = $paymentIntent->amount / 100; // Convert back to dollars
        $donorEmail = $paymentIntent->charges->data[0]->billing_details->email ?? 'Unknown';

        // Optional: Save donation to database
        $donationEntity = $this->Donations->newEntity([
            'amount' => $amount,
            'donor_email' => $donorEmail,
            'stripe_transaction_id' => $paymentIntent->id,
            'status' => 'completed'
        ]);

        $this->Donations->save($donationEntity);
    }
}
