<?php
declare(strict_types=1);

namespace App\Controller;

use App\Model\Table\StudentStatusesTable;
use Cake\I18n\FrozenTime;
use PhpOffice\PhpSpreadsheet\IOFactory;

/**
 * Students Controller
 *
 * @property \App\Model\Table\StudentsTable $Students
 */
class StudentsController extends AppController
{
    /**
     * Index method
     *
     * @return \Cake\Http\Response|null|void Renders view
     * @var \App\Model\Table\StudentStatusesTable $StudentStatusTable
     */
    private StudentStatusesTable $StudentStatusTable;
    public function initialize(): void
    {
        parent::initialize(); // TODO: Change the autogenerated stub
        $this->StudentStatusTable=$this->fetchTable('StudentStatuses');
    }

    public function index()
{
    $this->viewBuilder()->setLayout('adminlayout');

    // Fetch students and include their associated Programs and ProgramsStudents
    $query = $this->Students->find()
        ->contain(['StudentStatuses', 'Programs' => function ($q) {
            return $q->select(['Programs.id', 'Programs.name'])
                ->matching('ProgramsStudents');
        }]);


    // Fetch students
    $students = $this->paginate($query);

    // Loop through students and update their status based on associated programs
    foreach ($students as $student) {
        // Check if the student has associated programs and get the ProgramsStudents data
        $programStudentEntities = $this->Students->ProgramsStudents->find()
            ->where(['student_id' => $student->id])
            ->toArray();

        $atLeastOneInactive = false;
        foreach ($programStudentEntities as $programStudent) {
            // Check the status for each associated program
            if ($programStudent->status == 0) {
                $atLeastOneInactive = true;
                break;
            }
        }

        // Update the student status based on the program statuses
        if (empty($programStudentEntities)) {
            // If no programs are associated, set the student status to 'inactive' (3)
            $student->student_status_id = 3;
        } else {
            // If at least one program is inactive, set the student status to 'inactive' (2)
            $student->student_status_id = $atLeastOneInactive ? 2 : 1;
        }

        // Save the updated student status
        $this->Students->save($student);
    }

    // Pass the students to the view
    $this->set(compact('students'));
}




    /**
     * View method
     *
     * @param string|null $id Student id.
     * @return \Cake\Http\Response|null|void Renders view
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function view($id = null)
    {
        $student = $this->Students->get($id, contain: ['Programs']);
        $this->set(compact('student'));
    }

    /**
     * Add method
     *
     * @return \Cake\Http\Response|null|void Redirects on successful add, renders view otherwise.
     */
    public function add()
{

    $this->viewBuilder()->setLayout('adminlayout');

    $student = $this->Students->newEmptyEntity();

    if ($this->request->is('post')) {

        $contact_name = $this->request->getData('contact_name');
        $contact_email = $this->request->getData('contact_email');
        $contact_phone = $this->request->getData('contact_phone');

        $names = $this->request->getData('name');
        $years = $this->request->getData('year');
        $schools = $this->request->getData('school');
        $programs = $this->request->getData('programsData') ?? [];


        foreach ($programs as &$programsArray) {
            // Check if each $programsArray is a string (which could contain JSON data)
            if (is_string($programsArray[0])) {
                // Decode the string into an array (e.g., '[1,2]' to [1, 2])
                $programsArray = json_decode($programsArray[0], true);
            }
        }



        $students = [];
        $successCount = 0;
        $failureCount = 0;

        foreach ($names as $index => $name) {
            $student = $this->Students->newEmptyEntity();

            // Patch student entity with the form data
            $student = $this->Students->patchEntity($student, [
                'name' => $name,
                'year_level' => $years[$index],
                'school' => $schools[$index],
                'contact_name' => $contact_name,
                'contact_phone' => $contact_phone,
                'contact_email' => $contact_email,
                'registration_date' => null,
                'student_status_id' => 0
            ],['validate' => false]);

            // Get the programs selected for this particular student
            $selectedPrograms = isset($programs[$index]) ? $programs[$index] : [];

            // If programs are selected for the student, fetch them from the database
            if (!empty($selectedPrograms)) {
                $programEntities = $this->Students->Programs->find()
                    ->where(['id IN' => $selectedPrograms])
                    ->toArray(); // Convert ResultSet to array

                // Assign the selected programs to the student entity
                $student->programs = $programEntities;
            } else {
                // If no programs are selected, ensure no programs are assigned
                $student->programs = [];
            }

            // Save the student entity
            if ($this->Students->save($student)) {
                $students[] = $student;
                $successCount++;
            } else {
                $failureCount++;
            }
        }

        // Show success or error messages
        if ($successCount > 0) {
            $this->Flash->success(__(
                $successCount . ' student(s) have been registered successfully.'
            ));
            return $this->redirect(['action' => 'index']);
        }

        if ($failureCount > 0) {
            $this->Flash->error(__('There were ' . $failureCount . ' student(s) that could not be saved. Please, try again.'));
        }

    }

    // Fetch available programs to display in the form
    $programs = $this->Students->Programs->find('list')->limit(200)->all();
    $this->set(compact('programs', 'student'));
}



    /**
     * Edit method
     *
     * @param string|null $id Student id.
     * @return \Cake\Http\Response|null|void Redirects on successful edit, renders view otherwise.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function edit($id = null)
{
    $this->viewBuilder()->setLayout('adminlayout');
    $student = $this->Students->get($id, ['contain' => ['Programs']]);

    // Fetch the existing program associations
    $programStudentEntities = $this->Students->Programs->ProgramsStudents->find()
        ->where(['student_id' => $id])
        ->toArray();

    // Initialize program statuses array
    $programStatuses = [];

    // Set student status based on current program status
    $atLeastOneInactive = false;
    foreach ($programStudentEntities as $programStudent) {
        $programStatuses[$programStudent->program_id] = $programStudent->status;
        if ($programStudent->status == 0) {
            $atLeastOneInactive = true;
        }
    }

    // Set the student's initial status based on whether they have associated programs
    $student->student_status_id = empty($programStudentEntities) ? 3 : ($atLeastOneInactive ? 2 : 1);
    $this->Students->save($student);

    if ($this->request->is(['patch', 'post', 'put'])) {
        $student = $this->Students->patchEntity($student, $this->request->getData());

        $selectedPrograms = $this->request->getData('programs');  // Get selected programs
        $programStatusesForm = $this->request->getData('program_status');  // Get status of programs from form

        // If no programs are selected, delete all associations
        if (empty($selectedPrograms)) {
            // Remove all associated programs for this student
            $programsToRemove = $this->Students->Programs->ProgramsStudents->find()
                ->where(['student_id' => $student->id])
                ->toArray();

            foreach ($programsToRemove as $programStudent) {
                $this->Students->Programs->ProgramsStudents->delete($programStudent);
            }

            // Update the student status to "inactive" since there are no associated programs
            $student->student_status_id = 3;  // Status 3 could represent "inactive"

            if ($this->Students->save($student)) {
                $this->Flash->success(__('The student has been saved successfully.'));
                return $this->redirect(['action' => 'index']);
            } else {
                $this->Flash->error(__('The student could not be updated. Please, try again.'));
            }

        } else {
            // Process the selected programs
            $programEntities = $this->Students->Programs->find()
                ->where(['id IN' => $selectedPrograms])
                ->toArray();

            $student->programs = $programEntities;  // Assign the selected programs

            // Delete deselected associations
            $programsToRemove = $this->Students->Programs->ProgramsStudents->find()
                ->where(['student_id' => $student->id])
                ->toArray();

            foreach ($programsToRemove as $programStudent) {
                $programFound = false;
                foreach ($student->programs as $program) {
                    if ($programStudent->program_id == $program->id) {
                        $programFound = true;
                        break;
                    }
                }
                if (!$programFound) {
                    // Delete the programs the user no longer wants associated
                    $this->Students->Programs->ProgramsStudents->delete($programStudent);
                }
            }

            // Add newly selected programs if they are not already associated
            foreach ($student->programs as $program) {
                $existingAssociation = $this->Students->Programs->ProgramsStudents->find()
                    ->where([
                        'student_id' => $student->id,
                        'program_id' => $program->id
                    ])
                    ->first();

                if (!$existingAssociation) {
                    $programStudent = $this->Students->Programs->ProgramsStudents->newEntity([
                        'student_id' => $student->id,
                        'program_id' => $program->id,
                        'status' => 1,  // Default status is active (1)
                    ]);
                    $this->Students->Programs->ProgramsStudents->save($programStudent);
                }
            }

            // Update program status based on form input
            foreach ($student->programs as $program) {
                $programStudent = $this->Students->Programs->ProgramsStudents->find()
                    ->where([
                        'student_id' => $student->id,
                        'program_id' => $program->id
                    ])
                    ->first();

                if ($programStudent) {
                    $programStudent->status = isset($programStatusesForm[$program->id]) && $programStatusesForm[$program->id] == 'on' ? 1 : 0;
                    $this->Students->Programs->ProgramsStudents->save($programStudent);
                }
            }

            // Recheck the student status based on the updated program associations
            $programStudentEntities = $this->Students->Programs->ProgramsStudents->find()
                ->where(['student_id' => $student->id])
                ->toArray();


            if (empty($programStudentEntities)) {
                $student->student_status_id = 3;  // No programs left, set to inactive status
            } else {
                $atLeastOneInactive = false;
                foreach ($programStudentEntities as $programStudent) {
                    if ($programStudent->status == 0) {
                        $atLeastOneInactive = true;
                        break;
                    }
                }
                $student->student_status_id = $atLeastOneInactive ? 2 : 1;  // Status 2 could mean "partially active"
            }


            // Save the updated student
            if ($this->Students->save($student)) {
                $this->Flash->success(__('The student has been saved successfully.'));
                return $this->redirect(['action' => 'index']);
            } else {
                $this->Flash->error(__('The student could not be updated. Please, try again.'));
            }
        }
    }

    // Load the list of programs for the form
    $programs = $this->Students->Programs->find('list', ['limit' => 200])->all();
    $this->set(compact('student', 'programs', 'programStatuses'));
}

    public function export()
    {
        $start = $this->request->getQuery('start_date');
        $end   = $this->request->getQuery('end_date');

        $query = $this->Students->find()
            ->contain(['StudentStatuses'])
            ->select([
                'name',
                'year_level',
                'school',
                'contact_name',
                'contact_phone',
                'registration_date',
                'status' => 'StudentStatuses.name',
                'contact_email'
            ]);

        if ($start && $end) {
            $query->where([
                'Students.registration_date >=' => $start . ' 00:00:00',
                'Students.registration_date <=' => $end   . ' 23:59:59',
            ]);
        }

        $students = $query->toArray();

        $filename = sprintf(
            'students_%s_to_%s.csv',
            date('d-M-y', strtotime($start ?: 'now')),
            date('d-M-y', strtotime($end   ?: 'now'))
        );

        $this->response = $this->response->withDownload($filename);

        $headers = [
            'Name',
            'Year Level',
            'School',
            'Contact Name',
            'Contact Phone',
            'Registration Date',
            'Status',
            'Contact Email'
        ];

        $this->set(compact('students', 'headers'));
        $this->viewBuilder()
            ->setClassName('CsvView.Csv')
            ->setOption('serialize', 'students')
            ->setOption('header', $headers);
    }

    /**
     * Preview Students Export (paginated table).
     */
    public function preview()
    {
        $this->viewBuilder()->setLayout('adminlayout');
        $start = $this->request->getQuery('start_date');
        $end   = $this->request->getQuery('end_date');

        $query = $this->Students->find()
            ->contain(['StudentStatuses'])
            ->select([
                'name',
                'year_level',
                'school',
                'contact_name',
                'contact_phone',
                'registration_date',
                'status' => 'StudentStatuses.name',
                'contact_email'
            ])
            ->order(['Students.registration_date' => 'DESC']);

        if ($start && $end) {
            $query->where([
                'Students.registration_date >=' => "$start 00:00:00",
                'Students.registration_date <=' => "$end 23:59:59",
            ]);
        }

        // Paginate results for preview
        $students = $this->paginate($query);

        $this->set(compact('students', 'start', 'end'));
    }

    /**
     * Overview of Students with a filter form.
     */
    public function overview()
    {
        $this->viewBuilder()->setLayout('adminlayout');
        $start = $this->request->getQuery('start_date');
        $end   = $this->request->getQuery('end_date');

        $query = $this->Students->find()
            ->contain(['StudentStatuses'])
            ->order(['Students.registration_date' => 'DESC']);

        if ($start && $end) {
            $query->where([
                'Students.registration_date >=' => $start . ' 00:00:00',
                'Students.registration_date <=' => $end   . ' 23:59:59',
            ]);
        }

        $students = $this->paginate($query);

        $this->set(compact('students', 'start', 'end'));
    }

















    /**
     * Delete method
     *
     * @param string|null $id Student id.
     * @return \Cake\Http\Response|null Redirects to index.
     * @throws \Cake\Datasource\Exception\RecordNotFoundException When record not found.
     */
    public function delete($id = null)
    {
        $this->request->allowMethod(['post', 'delete']);
        $student = $this->Students->get($id);
        if ($this->Students->delete($student)) {
            $this->Flash->success(__('The student has been deleted.'));
        } else {
            $this->Flash->error(__('The student could not be deleted. Please, try again.'));
        }

        return $this->redirect(['action' => 'index']);
    }

    public function upload()
    {
        $this->viewBuilder()->setLayout('adminlayout'); // Keep layout consistent
        $insertedVolunteers = [];
        $failedVolunteers = [];
        $filepath = null; // Initialize filepath

        if ($this->request->is('post')) {
            $file = $this->request->getData('uploaded_file');

            if ($file && $file->getError() === UPLOAD_ERR_OK) {
                $filepath = $this->handleFileUpload($file);
                if (!$filepath) {
                    // Flash message is set in handleFileUpload
                    return $this->redirect(['action' => 'index']);
                }

                try {
                    // Check if it's a CSV to potentially adjust reader options
                    $fileExtension = strtolower(pathinfo($filepath, PATHINFO_EXTENSION));
                    if ($fileExtension === 'csv') {
                        // Optional: Set CSV specific reader options if needed
                        // $reader = IOFactory::createReader('Csv');
                        // $reader->setInputEncoding('UTF-8'); // Example: Force UTF-8
                        // $spreadsheet = $reader->load($filepath);
                        $spreadsheet = IOFactory::load($filepath); // IOFactory::load usually detects fine
                    } else {
                        $spreadsheet = IOFactory::load($filepath);
                    }

                    $worksheet = $spreadsheet->getActiveSheet();

                    if ($this->checkCsvHeader($worksheet) === false) { // Use new header check function
                        $this->Flash->error(__('Invalid or mismatched header row in the uploaded file. Please ensure the first row matches:
Name, Year Level ,School, Contact Name , Contact Phone, Registration Date, Status, Contact Email'));
                        // Clean up before redirecting
                        if ($filepath && file_exists($filepath)) {
                            unlink($filepath);
                        }
                        return $this->redirect(['action' => 'index']);
                    }

                    $this->processSpreadsheetData($worksheet, $insertedVolunteers, $failedVolunteers); // Use new processing function

                    // Store results in session for potential display (optional)
                    // $this->request->getSession()->write('insertedVolunteers', $insertedVolunteers);
                    // $this->request->getSession()->write('failedVolunteers', $failedVolunteers);

                    if (!empty($insertedVolunteers) || !empty($failedVolunteers)) {
                        $this->Flash->success(__('File processed. Generating report...'));
                        // Generate report only if there's something to report
                        $this->generatePdfReport($insertedVolunteers, $failedVolunteers); // Function handles exit/download
                        // Note: generatePdfReport calls exit(), so code below might not execute if PDF is generated
                    } else {
                        $this->Flash->warning(__('The uploaded file was empty or contained no processable data rows.'));
                    }
                    // Redirect occurs inside generatePdfReport or here if no report
                    return $this->redirect(['action' => 'index']);

                } catch (\PhpOffice\PhpSpreadsheet\Exception $e) { // Catch specific spreadsheet errors
                    $this->Flash->error(__('Error reading the spreadsheet file: ' . $e->getMessage()));
                } catch (\Exception $e) {
                    $this->Flash->error(__('An error occurred during processing: ' . $e->getMessage()));
                } finally {
                    // Clean up the uploaded file
                    if ($filepath && file_exists($filepath)) {
                        unlink($filepath);
                    }
                }
            } elseif ($file && $file->getError() !== UPLOAD_ERR_NO_FILE) {
                $this->Flash->error(__('Error during file upload: Code ' . $file->getError()));
            } else {
                $this->Flash->error(__('No file was uploaded. Please select a file.'));
            }
        } else {
            // If not POST, just render the index view which should contain the upload form
            // No need to redirect here, let the index action handle rendering
            // return $this->redirect(['action' => 'index']); // Avoid redirect loop
        }

        // Set variables for the view if needed (e.g., if rendering upload form directly)
        // $this->set(compact('insertedVolunteers', 'failedVolunteers'));
        // If the upload logic is on the index page, redirecting back to index is correct.
        // If 'upload' has its own view, remove the final redirect.
        // Assuming the form is on the index page:
        if (!$this->response->isDone()) { // Prevent headers already sent error if PDF was generated
            return $this->redirect(['action' => 'index']);
        }
    }

    private function processSpreadsheetData($worksheet, &$insertedVolunteers, &$failedVolunteers)
    {
        // Use toArray() for potentially easier row/column access, especially for CSV
        // null, true, true, true => read null values, calculate formulas, format values, read headings
        $dataArray = $worksheet->toArray('', true, true, false); // Get raw cell values

        // Remove header row
        array_shift($dataArray);

        foreach ($dataArray as $rowIndex => $row) {
            // $rowIndex is 0-based here after array_shift, representing actual row number - 2
            $actualRowNum = $rowIndex + 2; // For user feedback if needed
            $this->processVolunteerCsvRow($row, $actualRowNum, $insertedVolunteers, $failedVolunteers);
        }
    }

    // New expected header: ['Name', 'Year Level', 'School', 'Contact Name', 'Contact Phone', 'Registration Date', 'Status', 'Contact Email']
    private function processVolunteerCsvRow(array $row, int $rowNum, &$insertedVolunteers, &$failedVolunteers)
    {
        // Map new column indices based on your new header
        $name = $this->handleNullValues($row[0] ?? null);
        $yearLevel = $this->handleNullValues($row[1] ?? null);
        $school = $this->handleNullValues($row[2] ?? null);
        $contactName = $this->handleNullValues($row[3] ?? null);
        $contactPhone = $this->handleNullValues($row[4] ?? null);
        $registrationDate = $this->handleNullValues($row[5] ?? null);
        $student_status = $this->handleNullValues($row[6] ?? null);
        $contactEmail = $this->handleNullValues($row[7] ?? null);

        // Basic validation: Check if essential fields like contact email are present
        if (empty($contactEmail)) {
            $failedVolunteers[] = [
                'data' => compact('name', 'yearLevel', 'school', 'contactName', 'contactPhone', 'registrationDate', 'student_status', 'contactEmail'),
                'reason' => "Skipped row {$rowNum}: Contact email is missing.",
            ];
            return;
        }

        // Check for existing volunteer by contact email
        $existingVolunteer = $this->Students->find('all')->where(['contact_email' => $contactEmail])->first();
        if ($existingVolunteer) {
            $failedVolunteers[] = [
                'data' => compact('name', 'yearLevel', 'school', 'contactName', 'contactPhone', 'registrationDate', 'student_status', 'contactEmail'),
                'reason' => "Skipped row {$rowNum}: Contact email already exists.",
            ];
            return;
        }

        // Find volunteer status ID by name
        $statusId = null;
        if (!empty($student_status)) {
            $statusEntity = $this->StudentStatusTable->find()
                ->where(['name' => $student_status])
                ->first();
            if ($statusEntity) {
                $statusId = $statusEntity->id;
            } else {
                $failedVolunteers[] = [
                    'data' => compact('name', 'yearLevel', 'school', 'contactName', 'contactPhone', 'registrationDate', 'student_status', 'contactEmail'),
                    'reason' => "Skipped row {$rowNum}: Volunteer status '{$student_status}' not found.",
                ];
                return;
            }
        } else {
            $failedVolunteers[] = [
                'data' => compact('name', 'yearLevel', 'school', 'contactName', 'contactPhone', 'registrationDate', 'student_status', 'contactEmail'),
                'reason' => "Skipped row {$rowNum}: Volunteer status is missing.",
            ];
            return;
        }

        // Save data
        $result = $this->saveStudentData($name, $yearLevel, $school, $contactName, $contactPhone, $registrationDate, $statusId, $contactEmail);

        if ($result['status'] === 'success') {
            $insertedVolunteers[] = $result['volunteer'];
        } else {
            $failedVolunteers[] = [
                'data' => $result['data'],
                'reason' => "Failed to save row {$rowNum}: " . ($result['errors'] ? json_encode($result['errors']) : 'Unknown save error'),
            ];
        }
    }

    private function saveStudentData($name, $yearLevel, $school, $contactName, $contactPhone, $registrationDate, $student_status_id , $contactEmail)
    {
        $dataToSave = [
            'name' => $name,
            'first_name'=>'dldl',
             'last_name'=>'ddl',
            'email'=>$contactEmail,
            'phone'=>'dldl',
            'occupation'=>"dkdk",
            "description"=>"dmd",
            "postcode"=>123456,
            "phone"=>$contactPhone,
            'year_level' => $yearLevel,
            'school' => $school,
            'contact_name' => $contactName,
            'contact_phone' => $contactEmail === '' ? null : (int)preg_replace('/\D/', '', $contactPhone),
            'registration_date' => $registrationDate, // Should be properly formatted (e.g., 'Y-m-d')
            'student_status_id' => $student_status_id  === null ? null : (int)$student_status_id ,
            'contact_email' => $contactEmail,
        ];

        $student = $this->Students->newEntity($dataToSave); // Still using Volunteers table?

        if ($this->Students->save($student)) {
            return ['status' => 'success', 'volunteer' => $student, 'data' => $dataToSave, 'errors' => null];
        } else {
            \Cake\Log\Log::error('Student save failed: ' . print_r($student->getErrors(), true) . ' Data: ' . print_r($dataToSave, true));
            return ['status' => 'error', 'volunteer' => null, 'data' => $dataToSave, 'errors' => $student->getErrors()];
        }
    }




    private function checkCsvHeader($worksheet): bool
    {
        // Expected headers for CSV import
        $expectedHeader = ['Name', 'Year Level', 'School', 'Contact Name', 'Contact Phone', 'Registration Date', 'Status',"Contact Email"];

        $headerRow = 1;
        $actualHeader = [];
        $highestColumn = $worksheet->getHighestColumn(); // e.g., 'G'
        $highestColumnIndex = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::columnIndexFromString($highestColumn);

        for ($col = 1; $col <= $highestColumnIndex; $col++) {
            $cellValue = $worksheet->getCell(\PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($col) . $headerRow)->getValue();
            $actualHeader[] = trim((string)$cellValue);
        }


        // Compare only up to the number of expected headers
        $actualHeaderSlice = array_slice($actualHeader, 0, count($expectedHeader));
        return $actualHeaderSlice === $expectedHeader;
    }

    private function checkHeader($worksheet):bool{
//        $expectedHeader = "first_name\tlast_name\temail\tphone\tpostcode\toccupation\tvolunteer_status_id";

        $expectedHeader = "name\tyear_level\tschool\tcontact_name\tcontact_phone\tstudent_status\tcontact_email";

        // Get the header row as a string
        $headerRow = 1;
        $actualHeader = '';
        $columns = ['A', 'B', 'C', 'D', 'E', 'F', 'G'];

        foreach ($columns as $index => $column) {
            $cellValue = $worksheet->getCell($column . $headerRow)->getValue() ?? '';
            $actualHeader .= $cellValue;
            if ($index < count($columns) - 1) {
                $actualHeader .= "\t";
            }
        }
        $expectedHeader = preg_replace('/\s+/', '', $expectedHeader);  // Remove all whitespace (spaces, tabs, newlines) from expected header
        $actualHeader = preg_replace('/\s+/', '', $actualHeader);  // Remove all whitespace from actual header

//        \Cake\Error\debug($actualHeader);
//        \Cake\Error\debug($expectedHeader);

        // Validate headers
        if (trim($actualHeader) !== $expectedHeader) {
            return  false;
        }

//        $this->Flash->error($actualHeader);
        return true;
    }

    private function generatePdfReport($insertedStudents, $failedStudents)
    {
        ob_start();

        try {
            $pdf = new \TCPDF(PDF_PAGE_ORIENTATION, PDF_UNIT, PDF_PAGE_FORMAT, true, 'UTF-8', false);

            $pdf->SetCreator(PDF_CREATOR);
            $pdf->SetAuthor('Your Application');
            $pdf->SetTitle('Student Upload Report');
            $pdf->SetSubject('Upload Results');

            $pdf->setPrintHeader(false);
            $pdf->setPrintFooter(false);

            $pdf->SetMargins(15, 15, 15);
            $pdf->SetAutoPageBreak(true, 15);

            $pdf->AddPage();

            $html = '<h1>Student Upload Report</h1>';
            $html .= '<p>Date: ' . date('Y-m-d H:i:s') . '</p>';

            // Successfully Inserted Students
            $html .= '<h2>Successfully Uploaded Students (' . count($insertedStudents) . ')</h2>';
            if (!empty($insertedStudents)) {
                $html .= '<table border="1" cellpadding="5" cellspacing="0" style="border-collapse: collapse; width: 100%;">';
                $html .= '<tr style="background-color: #4CAF50; color: white;">';
                $html .= '<th>Name</th><th>Year Level</th><th>School</th><th>Contact Name</th>';
                $html .= '<th>Contact Phone</th><th>Registration Date</th><th>Status ID</th><th>Contact Email</th>';
                $html .= '</tr>';

                foreach ($insertedStudents as $student) {
                    $html .= '<tr>';
                    $html .= '<td>' . h($student->name) . '</td>';
                    $html .= '<td>' . h($student->year_level) . '</td>';
                    $html .= '<td>' . h($student->school) . '</td>';
                    $html .= '<td>' . h($student->contact_name) . '</td>';
                    $html .= '<td>' . h($student->contact_phone) . '</td>';
                    $html .= '<td>' . h($student->registration_date) . '</td>';
                    $html .= '<td>' . h($student->volunteer_status_id) . '</td>';
                    $html .= '<td>' . h($student->contact_email) . '</td>';
                    $html .= '</tr>';
                }
                $html .= '</table>';
            } else {
                $html .= '<p>No students were successfully uploaded.</p>';
            }

            // Failed Students
            $html .= '<h2>Failed Uploads (' . count($failedStudents) . ')</h2>';
            if (!empty($failedStudents)) {
                $html .= '<table border="1" cellpadding="5" cellspacing="0" style="border-collapse: collapse; width: 100%;">';
                $html .= '<tr style="background-color: #f44336; color: white;">';
                $html .= '<th>Name</th><th>Year Level</th><th>School</th><th>Contact Name</th>';
                $html .= '<th>Contact Phone</th><th>Registration Date</th><th>Status ID</th><th>Contact Email</th><th>Reason</th>';
                $html .= '</tr>';

                foreach ($failedStudents as $failed) {
                    $data = is_array($failed['data']) ? (object)$failed['data'] : $failed['data'];
                    $html .= '<tr>';
                    $html .= '<td>' . h($data->name ?? '') . '</td>';
                    $html .= '<td>' . h($data->year_level ?? '') . '</td>';
                    $html .= '<td>' . h($data->school ?? '') . '</td>';
                    $html .= '<td>' . h($data->contact_name ?? '') . '</td>';
                    $html .= '<td>' . h($data->contact_phone ?? '') . '</td>';
                    $html .= '<td>' . h($data->registration_date ?? '') . '</td>';
                    $html .= '<td>' . h($data->volunteer_status_id ?? '') . '</td>';
                    $html .= '<td>' . h($data->contact_email ?? '') . '</td>';
                    $html .= '<td>' . h($failed['reason']) . '</td>';
                    $html .= '</tr>';
                }
                $html .= '</table>';
            } else {
                $html .= '<p>No upload failures recorded.</p>';
            }

            $pdf->writeHTML($html, true, false, true, false, '');

            ob_end_clean();

            $filename = 'student_upload_report_' . date('Ymd_His') . '.pdf';
            $pdf->Output($filename, 'D');
            exit();
        } catch (\Exception $e) {
            ob_end_clean();
            $this->Flash->error('Error generating PDF: ' . $e->getMessage());
            return $this->redirect(['action' => 'index']);
        }
    }


    private function processStudentRow($worksheet, $row, &$insertedStudents, &$failedStudents)
    {
        $name = $worksheet->getCell('A' . $row)->getValue();
        $year_level = $worksheet->getCell('B' . $row)->getValue();
        $school = $worksheet->getCell('C' . $row)->getValue();
        $contact_name = $worksheet->getCell('D' . $row)->getValue();
        $contact_phone = $worksheet->getCell('E' . $row)->getValue();
        $student_status = $worksheet->getCell('F' . $row)->getValue();
        $contact_email = $this->handleNullValues($worksheet->getCell('G' . $row)->getValue());

        // Handle possible null values for required fields
        $name = $this->handleNullValues($name);
        $year_level = $this->handleNullValues($year_level);
        $school = $this->handleNullValues($school);
        $contact_name = $this->handleNullValues($contact_name);
        $contact_phone = $this->handleNullValues($contact_phone);
        $student_status = $this->handleNullValues($student_status);



        // Check if the student already exists by email
        $existingStudent = $this->Students->find('all')->where(['contact_email' => $contact_email])->first();
        $checkStatus = $this->StudentStatusTable->find('all')->where(['name' => $student_status])->first();
        if (!$existingStudent &&$checkStatus) {
            $statusId = $checkStatus->id;

            // Save the student data
            $result = $this->saveStudent($name, $year_level, $school, $contact_name, $contact_phone, $statusId, $contact_email);

            if ($result['status'] === 'success') {
                $insertedStudents[] = $result['student'];
            } else {
                $failedStudents[] = [
                    'data' => $result['student'],
                    'reason' => 'Failed to save'
                ];
            }
        } else {
            // Store all student data when email exists
            if ($existingStudent) {
                $failedStudents[] = [
                    'data' => [
                        'name' => $name,
                        'year_level' => $year_level,
                        'school' => $school,
                        'contact_name' => $contact_name,
                        'contact_phone' => $contact_phone,
                        'student_status' => $student_status,
                        'contact_email' => $contact_email
                    ],
                    'reason' => 'Email already in use'
                ];
            }else{
                $failedStudents[] = [
                    'data' => [
                        'name' => $name,
                        'year_level' => $year_level,
                        'school' => $school,
                        'contact_name' => $contact_name,
                        'contact_phone' => $contact_phone,
                        'student_status' => $student_status,
                        'contact_email' => $contact_email
                    ],
                    'reason' => 'Invalid student status'
                ];
            }
        }
    }

    private function saveStudent($name, $year_level, $school, $contact_name, $contact_phone, $student_status_id, $contact_email)
    {
        // Create a new student entity
        $student = $this->Students->newEntity([
            'name' => $name,
            'year_level' => $year_level,
            'school' => $school,
            'contact_name' => $contact_name,
            'contact_phone' => $contact_phone === '' ? null : (int) $contact_phone,
            'student_status_id' => $student_status_id === '' ? null : (int) $student_status_id,
            'contact_email' => $contact_email === '' ? null : $contact_email,
        ]);

        // Uncomment the following block to debug if there are errors while saving
        // if ($student->getErrors()) {
        //     \Cake\Error\debug($student->getErrors());
        // }

        if ($this->Students->save($student)) {
            return ['status' => 'success', 'student' => $student];
        } else {
            return ['status' => 'error', 'student' => $student];
        }
    }



    private function processSpreadsheet($worksheet, &$insertedVolunteers, &$failedVolunteers)
    {
        $highestRow = $worksheet->getHighestRow();
        for ($row = 2; $row <= $highestRow; $row++) {
            $this->processStudentRow($worksheet, $row, $insertedVolunteers, $failedVolunteers);
        }
    }

    private function handleFileUpload($file)
    {
        $allowedTypes = [
            'application/vnd.ms-excel',
            'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
            'text/csv'
        ];

        if (in_array($file->getClientMediaType(), $allowedTypes)) {
            $filename = $file->getClientFilename();
            $filepath = WWW_ROOT . 'uploads' . DS . time() . '-' . $filename;
            $file->moveTo($filepath);
            return $filepath;
        } else {
            $this->Flash->error(__('Invalid file type.'));
            return false;
        }
    }


    private function handleNullValues($value)
    {
        return $value ?? '';
    }


}
