<?php
declare(strict_types=1);

namespace App\Model\Table;

use Cake\ORM\RulesChecker;
use Cake\ORM\Table;
use Cake\Validation\Validator;

/**
 * Programs Model
 *
 * @property \App\Model\Table\ProgramStatusesTable&\Cake\ORM\Association\BelongsTo $ProgramStatuses
 * @property \App\Model\Table\ProgramsImagesTable&\Cake\ORM\Association\HasMany $ProgramsImages
 * @property \App\Model\Table\StudentsTable&\Cake\ORM\Association\BelongsToMany $Students
 * @property \App\Model\Table\VolunteersTable&\Cake\ORM\Association\BelongsToMany $Volunteers
 */
class ProgramsTable extends Table
{
    /**
     * Initialize method
     *
     * @param array<string, mixed> $config The configuration for the Table.
     * @return void
     */
    public function initialize(array $config): void
    {
        parent::initialize($config);

        $this->setTable('programs');
        $this->setDisplayField('name');
        $this->setPrimaryKey('id');

        $this->belongsTo('ProgramStatuses', [
            'foreignKey' => 'program_status_id',
        ]);
        $this->hasMany('ProgramsImages', [
            'foreignKey' => 'program_id',
        ]);
        $this->belongsToMany('Students', [
            'foreignKey' => 'program_id',
            'targetForeignKey' => 'student_id',
            'joinTable' => 'programs_students',
            'cascadeCallbacks' => true,
        ]);
        $this->belongsToMany('Volunteers', [
            'foreignKey' => 'program_id',
            'targetForeignKey' => 'volunteer_id',
            'joinTable' => 'programs_volunteers',
            'cascadeCallbacks' => true,
        ]);
    }

    /**
     * Default validation rules.
     *
     * @param \Cake\Validation\Validator $validator Validator instance.
     * @return \Cake\Validation\Validator
     */
    public function validationDefault(Validator $validator): Validator
{
    // Validate the 'name' field
    $validator
        ->requirePresence('name')
        ->notEmptyString('name')
        ->add('name', 'validChars', [
            'rule' => ['custom', '/^[a-zA-Z0-9\s]+$/'],
            'message' => 'Only letters,numbers and spaces allowed.'
        ])
        ->maxLength('name', 100, 'Name cannot be longer than 100 characters');



    // Validate the 'date' field: required, valid date, not in the past
    $validator
        ->date('date')
        ->requirePresence('date', 'create')
        ->notEmptyDate('date', 'Please select a valid date')
        ->add('date', 'futureDate', [
            'rule' => function ($value) {
                return (new \DateTime($value))->format('Y-m-d') >= (new \DateTime())->format('Y-m-d');
            },
            'message' => 'Date cannot be in the past'
        ]);

    // Validate the 'time' field: required only, no custom restriction
    $validator
        ->time('time')
        ->requirePresence('time', 'create')
        ->notEmptyTime('time', 'Please enter a valid time');

    // Validate the 'program_status_id' field
    $validator
        ->integer('program_status_id')
        ->allowEmptyString('program_status_id');

    return $validator;
}

    
    /**
     * Returns a rules checker object that will be used for validating
     * application integrity.
     *
     * @param \Cake\ORM\RulesChecker $rules The rules object to be modified.
     * @return \Cake\ORM\RulesChecker
     */
    public function buildRules(RulesChecker $rules): RulesChecker
    {
        $rules->add($rules->existsIn(['program_status_id'], 'ProgramStatuses'), ['errorField' => 'program_status_id']);

        return $rules;
    }
}
